import { Link, useLocation } from "wouter";
import { ShoppingCart, User, Search, Menu, X, Moon, Sun, ChevronDown, Home, LayoutGrid, Phone } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import { useCart } from "@/lib/cart";
import { useAuth } from "@/lib/auth";
import { useTheme } from "@/lib/theme";
import { useState } from "react";
import { useQuery } from "@tanstack/react-query";
import type { Category } from "@shared/schema";

export default function Header() {
  const { totalItems } = useCart();
  const { user } = useAuth();
  const { theme, toggleTheme } = useTheme();
  const [searchOpen, setSearchOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState("");
  const [, navigate] = useLocation();
  const [mobileOpen, setMobileOpen] = useState(false);

  const { data: categories } = useQuery<Category[]>({
    queryKey: ["/api/categories"],
  });

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    if (searchQuery.trim()) {
      navigate(`/products?search=${encodeURIComponent(searchQuery.trim())}`);
      setSearchOpen(false);
      setSearchQuery("");
    }
  };

  return (
    <header className="sticky top-0 z-50 bg-background/95 backdrop-blur-sm border-b">
      <div className="max-w-7xl mx-auto px-4">
        <div className="flex items-center justify-between gap-4 h-16">
          <div className="flex items-center gap-3">
            <Sheet open={mobileOpen} onOpenChange={setMobileOpen}>
              <SheetTrigger asChild>
                <Button size="icon" variant="ghost" className="md:hidden" data-testid="button-mobile-menu">
                  <Menu />
                </Button>
              </SheetTrigger>
              <SheetContent side="right" className="w-80 p-0">
                <div className="p-4 border-b">
                  <h2 className="text-lg font-bold" data-testid="text-menu-title">منو</h2>
                </div>
                <nav className="p-4 space-y-1">
                  <Link href="/" onClick={() => setMobileOpen(false)}>
                    <div className="flex items-center gap-3 p-3 rounded-md hover-elevate cursor-pointer" data-testid="link-mobile-home">
                      <Home className="w-5 h-5" />
                      <span>صفحه اصلی</span>
                    </div>
                  </Link>
                  <Link href="/products" onClick={() => setMobileOpen(false)}>
                    <div className="flex items-center gap-3 p-3 rounded-md hover-elevate cursor-pointer" data-testid="link-mobile-products">
                      <LayoutGrid className="w-5 h-5" />
                      <span>محصولات</span>
                    </div>
                  </Link>
                  {categories?.map((cat) => (
                    <Link key={cat.id} href={`/products?category=${cat.slug}`} onClick={() => setMobileOpen(false)}>
                      <div className="flex items-center gap-3 p-3 pr-10 rounded-md hover-elevate cursor-pointer text-sm text-muted-foreground" data-testid={`link-mobile-cat-${cat.id}`}>
                        <span>{cat.name}</span>
                      </div>
                    </Link>
                  ))}
                  <Link href="/contact" onClick={() => setMobileOpen(false)}>
                    <div className="flex items-center gap-3 p-3 rounded-md hover-elevate cursor-pointer" data-testid="link-mobile-contact">
                      <Phone className="w-5 h-5" />
                      <span>تماس با ما</span>
                    </div>
                  </Link>
                </nav>
              </SheetContent>
            </Sheet>

            <Link href="/">
              <span className="text-xl font-bold text-primary cursor-pointer" data-testid="link-logo">فروشگاه</span>
            </Link>
          </div>

          <nav className="hidden md:flex items-center gap-1">
            <Link href="/">
              <Button variant="ghost" data-testid="link-home">صفحه اصلی</Button>
            </Link>
            <div className="relative group">
              <Button variant="ghost" className="gap-1" data-testid="button-categories-dropdown">
                دسته‌بندی‌ها
                <ChevronDown className="w-4 h-4" />
              </Button>
              <div className="absolute top-full right-0 w-56 bg-popover border rounded-md opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200 z-50">
                <div className="p-2">
                  <Link href="/products">
                    <div className="p-2 rounded-md hover-elevate cursor-pointer text-sm" data-testid="link-all-products">همه محصولات</div>
                  </Link>
                  {categories?.map((cat) => (
                    <Link key={cat.id} href={`/products?category=${cat.slug}`}>
                      <div className="p-2 rounded-md hover-elevate cursor-pointer text-sm" data-testid={`link-cat-${cat.id}`}>{cat.name}</div>
                    </Link>
                  ))}
                </div>
              </div>
            </div>
            <Link href="/contact">
              <Button variant="ghost" data-testid="link-contact">تماس با ما</Button>
            </Link>
          </nav>

          <div className="flex items-center gap-1">
            {searchOpen ? (
              <form onSubmit={handleSearch} className="flex items-center gap-2">
                <Input
                  type="search"
                  placeholder="جستجو..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="w-40 md:w-60"
                  autoFocus
                  data-testid="input-search"
                />
                <Button size="icon" variant="ghost" type="button" onClick={() => setSearchOpen(false)} data-testid="button-close-search">
                  <X className="w-4 h-4" />
                </Button>
              </form>
            ) : (
              <Button size="icon" variant="ghost" onClick={() => setSearchOpen(true)} data-testid="button-search">
                <Search className="w-5 h-5" />
              </Button>
            )}

            <Button size="icon" variant="ghost" onClick={toggleTheme} data-testid="button-theme-toggle">
              {theme === "light" ? <Moon className="w-5 h-5" /> : <Sun className="w-5 h-5" />}
            </Button>

            <Link href={user ? "/account" : "/login"}>
              <Button size="icon" variant="ghost" data-testid="button-account">
                <User className="w-5 h-5" />
              </Button>
            </Link>

            <Link href="/cart">
              <Button size="icon" variant="ghost" className="relative" data-testid="button-cart">
                <ShoppingCart className="w-5 h-5" />
                {totalItems > 0 && (
                  <span className="absolute -top-1 -right-1 bg-primary text-primary-foreground text-xs rounded-full w-5 h-5 flex items-center justify-center font-bold" data-testid="text-cart-count">
                    {totalItems}
                  </span>
                )}
              </Button>
            </Link>
          </div>
        </div>
      </div>
    </header>
  );
}
