import { useLocation } from "wouter";
import { ShoppingCart } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { formatPrice } from "@/lib/format";
import { useCart } from "@/lib/cart";
import type { Product } from "@shared/schema";

interface ProductCardProps {
  product: Product;
}

export default function ProductCard({ product }: ProductCardProps) {
  const { addItem } = useCart();
  const [, navigate] = useLocation();
  const discount = product.salePrice
    ? Math.round(((Number(product.price) - Number(product.salePrice)) / Number(product.price)) * 100)
    : 0;

  const goToProduct = () => navigate(`/products/${product.slug}`);

  return (
    <Card className="group overflow-visible relative hover-elevate cursor-pointer" data-testid={`card-product-${product.id}`} onClick={goToProduct}>
      <div className="relative aspect-square overflow-hidden rounded-t-md">
        <img
          src={product.images?.[0] || "/images/placeholder.png"}
          alt={product.name}
          className="w-full h-full object-cover"
          data-testid={`img-product-${product.id}`}
        />
        {discount > 0 && (
          <Badge className="absolute top-2 right-2 bg-destructive text-destructive-foreground" data-testid={`badge-discount-${product.id}`}>
            {discount}%-
          </Badge>
        )}
        {!product.inStock && (
          <div className="absolute inset-0 bg-background/70 flex items-center justify-center">
            <Badge variant="secondary" data-testid={`badge-outofstock-${product.id}`}>ناموجود</Badge>
          </div>
        )}
      </div>

      <div className="p-3 space-y-2">
        <h3
          className="font-medium text-sm line-clamp-2 leading-relaxed"
          data-testid={`text-product-name-${product.id}`}
        >
          {product.name}
        </h3>

        <div className="flex items-center justify-between gap-2 flex-wrap">
          <div className="flex flex-col">
            {product.salePrice ? (
              <>
                <span className="text-xs text-muted-foreground line-through" data-testid={`text-original-price-${product.id}`}>
                  {formatPrice(product.price)}
                </span>
                <span className="text-sm font-bold text-primary" data-testid={`text-sale-price-${product.id}`}>
                  {formatPrice(product.salePrice)}
                </span>
              </>
            ) : (
              <span className="text-sm font-bold" data-testid={`text-price-${product.id}`}>
                {formatPrice(product.price)}
              </span>
            )}
          </div>

          {product.inStock && (
            <Button
              size="icon"
              variant="ghost"
              className="rounded-full"
              onClick={(e) => {
                e.stopPropagation();
                addItem(product);
              }}
              data-testid={`button-add-cart-${product.id}`}
            >
              <ShoppingCart className="w-4 h-4" />
            </Button>
          )}
        </div>
      </div>
    </Card>
  );
}
