import { useQuery } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { User, Package, MapPin, LogOut, Settings, ChevronLeft } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Separator } from "@/components/ui/separator";
import { Skeleton } from "@/components/ui/skeleton";
import { useAuth } from "@/lib/auth";
import { formatPrice, formatDate, orderStatusMap } from "@/lib/format";
import { useToast } from "@/hooks/use-toast";
import type { Order, OrderItem } from "@shared/schema";
import { useState } from "react";

export default function Account() {
  const { user, logout, updateProfile, isLoading: authLoading } = useAuth();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [saving, setSaving] = useState(false);

  const [form, setForm] = useState({
    name: user?.name || "",
    email: user?.email || "",
    address: user?.address || "",
    city: user?.city || "",
    postalCode: user?.postalCode || "",
  });

  const { data: orders, isLoading: ordersLoading } = useQuery<(Order & { items: OrderItem[] })[]>({
    queryKey: ["/api/orders"],
    enabled: !!user,
  });

  if (authLoading) {
    return (
      <div className="max-w-4xl mx-auto px-4 py-6">
        <Skeleton className="h-8 w-48 mb-6" />
        <Skeleton className="h-64 w-full" />
      </div>
    );
  }

  if (!user) {
    navigate("/login");
    return null;
  }

  const handleSave = async (e: React.FormEvent) => {
    e.preventDefault();
    setSaving(true);
    try {
      await updateProfile(form);
      toast({ title: "پروفایل به‌روزرسانی شد" });
    } catch (err: any) {
      toast({ title: "خطا", description: err.message, variant: "destructive" });
    } finally {
      setSaving(false);
    }
  };

  const handleLogout = () => {
    logout();
    navigate("/");
  };

  return (
    <div className="max-w-4xl mx-auto px-4 py-6">
      <div className="flex items-center justify-between mb-6 flex-wrap gap-3">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-account-title">حساب کاربری</h1>
          <p className="text-sm text-muted-foreground" data-testid="text-account-phone">{user.phone}</p>
        </div>
        <Button variant="outline" size="sm" onClick={handleLogout} className="gap-1" data-testid="button-logout">
          <LogOut className="w-4 h-4" />
          خروج
        </Button>
      </div>

      <Tabs defaultValue="orders" dir="rtl">
        <TabsList className="mb-4">
          <TabsTrigger value="orders" className="gap-1" data-testid="tab-orders">
            <Package className="w-4 h-4" />
            سفارشات
          </TabsTrigger>
          <TabsTrigger value="profile" className="gap-1" data-testid="tab-profile">
            <User className="w-4 h-4" />
            پروفایل
          </TabsTrigger>
          <TabsTrigger value="addresses" className="gap-1" data-testid="tab-addresses">
            <MapPin className="w-4 h-4" />
            آدرس
          </TabsTrigger>
        </TabsList>

        <TabsContent value="orders">
          {ordersLoading ? (
            <div className="space-y-3">
              {Array.from({ length: 3 }).map((_, i) => (
                <Skeleton key={i} className="h-24 w-full rounded-md" />
              ))}
            </div>
          ) : orders && orders.length > 0 ? (
            <div className="space-y-3">
              {orders.map((order) => {
                const statusInfo = orderStatusMap[order.status] || orderStatusMap.pending;
                return (
                  <Card key={order.id} className="p-4" data-testid={`card-order-${order.id}`}>
                    <div className="flex items-center justify-between gap-3 mb-3 flex-wrap">
                      <div className="flex items-center gap-3">
                        <span className="text-sm font-medium" data-testid={`text-order-id-${order.id}`}>سفارش #{order.id}</span>
                        <Badge className={statusInfo.color} data-testid={`badge-order-status-${order.id}`}>
                          {statusInfo.label}
                        </Badge>
                      </div>
                      <span className="text-xs text-muted-foreground" data-testid={`text-order-date-${order.id}`}>
                        {formatDate(order.createdAt)}
                      </span>
                    </div>

                    {order.items && order.items.length > 0 && (
                      <div className="space-y-2 mb-3">
                        {order.items.map((item) => (
                          <div key={item.id} className="flex items-center gap-3 text-sm">
                            {item.productImage && (
                              <img src={item.productImage} alt="" className="w-10 h-10 rounded-md object-cover" />
                            )}
                            <span className="flex-1 line-clamp-1">{item.productName}</span>
                            <span className="text-muted-foreground">x{item.quantity}</span>
                            <span>{formatPrice(Number(item.price) * item.quantity)}</span>
                          </div>
                        ))}
                      </div>
                    )}

                    <Separator />
                    <div className="flex items-center justify-between mt-3 gap-2">
                      <span className="text-sm text-muted-foreground">جمع کل:</span>
                      <span className="font-bold text-primary" data-testid={`text-order-total-${order.id}`}>{formatPrice(order.total)}</span>
                    </div>
                  </Card>
                );
              })}
            </div>
          ) : (
            <Card className="p-12 text-center">
              <Package className="w-12 h-12 mx-auto text-muted-foreground mb-3" />
              <p className="text-muted-foreground" data-testid="text-no-orders">هنوز سفارشی ثبت نکرده‌اید</p>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="profile">
          <Card className="p-5">
            <form onSubmit={handleSave} className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="name">نام و نام خانوادگی</Label>
                  <Input
                    id="name"
                    value={form.name}
                    onChange={(e) => setForm({ ...form, name: e.target.value })}
                    data-testid="input-profile-name"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="email">ایمیل</Label>
                  <Input
                    id="email"
                    type="email"
                    value={form.email}
                    onChange={(e) => setForm({ ...form, email: e.target.value })}
                    dir="ltr"
                    data-testid="input-profile-email"
                  />
                </div>
              </div>
              <Button type="submit" disabled={saving} data-testid="button-save-profile">
                {saving ? "ذخیره..." : "ذخیره تغییرات"}
              </Button>
            </form>
          </Card>
        </TabsContent>

        <TabsContent value="addresses">
          <Card className="p-5">
            <form onSubmit={handleSave} className="space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="city">شهر</Label>
                  <Input
                    id="city"
                    value={form.city}
                    onChange={(e) => setForm({ ...form, city: e.target.value })}
                    data-testid="input-profile-city"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="postalCode">کد پستی</Label>
                  <Input
                    id="postalCode"
                    value={form.postalCode}
                    onChange={(e) => setForm({ ...form, postalCode: e.target.value })}
                    dir="ltr"
                    data-testid="input-profile-postal"
                  />
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="address">آدرس کامل</Label>
                <Input
                  id="address"
                  value={form.address}
                  onChange={(e) => setForm({ ...form, address: e.target.value })}
                  data-testid="input-profile-address"
                />
              </div>
              <Button type="submit" disabled={saving} data-testid="button-save-address">
                {saving ? "ذخیره..." : "ذخیره آدرس"}
              </Button>
            </form>
          </Card>
        </TabsContent>
      </Tabs>
    </div>
  );
}
