import { useQuery, useMutation } from "@tanstack/react-query";
import { useState, useEffect } from "react";
import { useLocation } from "wouter";
import { Package, ShoppingCart, LayoutGrid, ArrowLeft, Save, RefreshCw, Eye, EyeOff, ArrowUp, ArrowDown, Globe, Link2, CheckCircle, AlertCircle, Clock } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { Separator } from "@/components/ui/separator";
import { Skeleton } from "@/components/ui/skeleton";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { useAuth } from "@/lib/auth";
import { formatPrice, formatDate, orderStatusMap } from "@/lib/format";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import type { Product, Category, Order, OrderItem } from "@shared/schema";

interface SyncStatus {
  lastSync: string | null;
  syncResult: string | null;
  productCount: number;
  categoryCount: number;
}

export default function Admin() {
  const { user } = useAuth();
  const [, navigate] = useLocation();
  const { toast } = useToast();

  const { data: products, isLoading: productsLoading } = useQuery<Product[]>({
    queryKey: ["/api/admin/products"],
    enabled: !!user,
  });

  const { data: categories } = useQuery<Category[]>({
    queryKey: ["/api/admin/categories"],
    enabled: !!user,
  });

  const { data: orders, isLoading: ordersLoading } = useQuery<(Order & { items: OrderItem[] })[]>({
    queryKey: ["/api/admin/orders"],
    enabled: !!user,
  });

  const { data: syncStatus } = useQuery<SyncStatus>({
    queryKey: ["/api/admin/sync-status"],
    enabled: !!user,
  });

  const syncMutation = useMutation({
    mutationFn: () => apiRequest("POST", "/api/admin/sync"),
    onSuccess: async (res: any) => {
      const data = await res.json();
      queryClient.invalidateQueries({ queryKey: ["/api/products"] });
      queryClient.invalidateQueries({ queryKey: ["/api/categories"] });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/products"] });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/categories"] });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/sync-status"] });
      toast({ title: "همگام‌سازی موفق", description: data.message });
    },
    onError: (err: any) => {
      toast({ title: "خطا در همگام‌سازی", description: err.message, variant: "destructive" });
    },
  });

  const updateOrderStatus = useMutation({
    mutationFn: ({ id, status }: { id: number; status: string }) =>
      apiRequest("PATCH", `/api/admin/orders/${id}`, { status }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/orders"] });
      toast({ title: "وضعیت سفارش به‌روزرسانی شد" });
    },
  });

  if (!user || user.role !== "admin") {
    return (
      <div className="max-w-md mx-auto px-4 py-16 text-center">
        <h1 className="text-xl font-bold mb-2" data-testid="text-admin-denied">دسترسی غیرمجاز</h1>
        <p className="text-muted-foreground mb-4">شما اجازه دسترسی به این بخش را ندارید.</p>
        <Button onClick={() => navigate("/")} data-testid="button-admin-home">بازگشت به صفحه اصلی</Button>
      </div>
    );
  }

  return (
    <div className="max-w-7xl mx-auto px-4 py-6">
      <div className="flex items-center justify-between mb-6 flex-wrap gap-3">
        <h1 className="text-2xl font-bold" data-testid="text-admin-title">پنل مدیریت</h1>
        <div className="flex items-center gap-2">
          <Button
            variant="outline"
            size="sm"
            onClick={() => syncMutation.mutate()}
            disabled={syncMutation.isPending}
            className="gap-1"
            data-testid="button-admin-sync"
          >
            <RefreshCw className={`w-4 h-4 ${syncMutation.isPending ? "animate-spin" : ""}`} />
            {syncMutation.isPending ? "در حال همگام‌سازی..." : "همگام‌سازی با وردپرس"}
          </Button>
          <Button onClick={() => navigate("/")} variant="ghost" size="sm" className="gap-1" data-testid="button-admin-back">
            <ArrowLeft className="w-4 h-4" />
            سایت
          </Button>
        </div>
      </div>

      {syncStatus?.lastSync && (
        <Card className="p-3 mb-4" data-testid="card-sync-status">
          <div className="flex items-center gap-3 flex-wrap text-sm">
            <div className="flex items-center gap-1 text-muted-foreground">
              <Clock className="w-4 h-4" />
              آخرین همگام‌سازی: {formatDate(syncStatus.lastSync)}
            </div>
            {syncStatus.syncResult && (
              <div className="flex items-center gap-1">
                <CheckCircle className="w-4 h-4 text-green-600 dark:text-green-400" />
                {syncStatus.syncResult}
              </div>
            )}
          </div>
        </Card>
      )}

      <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
        <Card className="p-4" data-testid="card-admin-stat-products">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-primary/10 flex items-center justify-center">
              <Package className="w-5 h-5 text-primary" />
            </div>
            <div>
              <p className="text-2xl font-bold">{products?.length || 0}</p>
              <p className="text-xs text-muted-foreground">محصول</p>
            </div>
          </div>
        </Card>
        <Card className="p-4" data-testid="card-admin-stat-orders">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-blue-100 dark:bg-blue-900/30 flex items-center justify-center">
              <ShoppingCart className="w-5 h-5 text-blue-600 dark:text-blue-400" />
            </div>
            <div>
              <p className="text-2xl font-bold">{orders?.length || 0}</p>
              <p className="text-xs text-muted-foreground">سفارش</p>
            </div>
          </div>
        </Card>
        <Card className="p-4" data-testid="card-admin-stat-cats">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-green-100 dark:bg-green-900/30 flex items-center justify-center">
              <LayoutGrid className="w-5 h-5 text-green-600 dark:text-green-400" />
            </div>
            <div>
              <p className="text-2xl font-bold">{categories?.length || 0}</p>
              <p className="text-xs text-muted-foreground">دسته‌بندی</p>
            </div>
          </div>
        </Card>
        <Card className="p-4" data-testid="card-admin-stat-revenue">
          <div className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-full bg-purple-100 dark:bg-purple-900/30 flex items-center justify-center">
              <Globe className="w-5 h-5 text-purple-600 dark:text-purple-400" />
            </div>
            <div>
              <p className="text-2xl font-bold">
                {orders ? formatPrice(orders.reduce((s, o) => s + Number(o.total), 0)) : "0"}
              </p>
              <p className="text-xs text-muted-foreground">فروش کل</p>
            </div>
          </div>
        </Card>
      </div>

      <Tabs defaultValue="products" dir="rtl">
        <TabsList className="mb-4">
          <TabsTrigger value="products" data-testid="tab-admin-products">محصولات</TabsTrigger>
          <TabsTrigger value="categories" data-testid="tab-admin-categories">دسته‌بندی‌ها</TabsTrigger>
          <TabsTrigger value="orders" data-testid="tab-admin-orders">سفارشات</TabsTrigger>
          <TabsTrigger value="settings" data-testid="tab-admin-settings">تنظیمات ووکامرس</TabsTrigger>
        </TabsList>

        <TabsContent value="products">
          <div className="flex items-center justify-between mb-4 gap-3 flex-wrap">
            <h2 className="font-bold" data-testid="text-admin-products-list">محصولات (از ووکامرس)</h2>
            <p className="text-sm text-muted-foreground">محصولات از سایت وردپرسی شما خوانده می‌شوند. برای به‌روزرسانی، دکمه همگام‌سازی را بزنید.</p>
          </div>

          {productsLoading ? (
            <div className="space-y-2">
              {Array.from({ length: 5 }).map((_, i) => <Skeleton key={i} className="h-12 w-full" />)}
            </div>
          ) : products && products.length > 0 ? (
            <Card className="overflow-hidden">
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>تصویر</TableHead>
                    <TableHead>نام</TableHead>
                    <TableHead>قیمت</TableHead>
                    <TableHead>موجودی</TableHead>
                    <TableHead>دسته‌بندی</TableHead>
                    <TableHead>وضعیت</TableHead>
                    <TableHead>منبع</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {products.map((p) => (
                    <TableRow key={p.id} data-testid={`row-admin-product-${p.id}`}>
                      <TableCell>
                        <img src={p.images?.[0] || "/images/placeholder.png"} alt="" className="w-10 h-10 rounded-md object-cover" />
                      </TableCell>
                      <TableCell className="font-medium text-sm max-w-[200px] truncate">{p.name}</TableCell>
                      <TableCell className="text-sm">{formatPrice(p.salePrice || p.price)}</TableCell>
                      <TableCell className="text-sm">{p.stockQuantity}</TableCell>
                      <TableCell className="text-sm">
                        {categories?.find(c => c.id === p.categoryId)?.name || "-"}
                      </TableCell>
                      <TableCell>
                        <Badge variant={p.inStock ? "default" : "secondary"}>
                          {p.inStock ? "موجود" : "ناموجود"}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        {p.wcId ? (
                          <Badge variant="outline" className="gap-1">
                            <Link2 className="w-3 h-3" />
                            WC #{p.wcId}
                          </Badge>
                        ) : (
                          <Badge variant="secondary">محلی</Badge>
                        )}
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </Card>
          ) : (
            <Card className="p-8 text-center">
              <Package className="w-12 h-12 mx-auto mb-3 text-muted-foreground" />
              <h3 className="font-bold mb-2">هنوز محصولی همگام‌سازی نشده</h3>
              <p className="text-sm text-muted-foreground mb-4">ابتدا تنظیمات ووکامرس را وارد کنید و سپس دکمه همگام‌سازی را بزنید.</p>
              <Button onClick={() => syncMutation.mutate()} disabled={syncMutation.isPending} className="gap-1" data-testid="button-empty-sync">
                <RefreshCw className={`w-4 h-4 ${syncMutation.isPending ? "animate-spin" : ""}`} />
                همگام‌سازی
              </Button>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="categories">
          <CategoryManager categories={categories || []} />
        </TabsContent>

        <TabsContent value="orders">
          {ordersLoading ? (
            <div className="space-y-2">
              {Array.from({ length: 5 }).map((_, i) => <Skeleton key={i} className="h-16 w-full" />)}
            </div>
          ) : orders && orders.length > 0 ? (
            <div className="space-y-3">
              {orders.map((order) => {
                const statusInfo = orderStatusMap[order.status] || orderStatusMap.pending;
                return (
                  <Card key={order.id} className="p-4" data-testid={`card-admin-order-${order.id}`}>
                    <div className="flex items-center justify-between gap-3 flex-wrap mb-3">
                      <div className="flex items-center gap-3 flex-wrap">
                        <span className="font-medium text-sm">سفارش #{order.id}</span>
                        <Badge className={statusInfo.color}>{statusInfo.label}</Badge>
                        <span className="text-xs text-muted-foreground">{formatDate(order.createdAt)}</span>
                      </div>
                      <Select
                        value={order.status}
                        onValueChange={(v) => updateOrderStatus.mutate({ id: order.id, status: v })}
                      >
                        <SelectTrigger className="w-36 text-sm" data-testid={`select-admin-status-${order.id}`}>
                          <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="pending">در انتظار پرداخت</SelectItem>
                          <SelectItem value="processing">در حال پردازش</SelectItem>
                          <SelectItem value="shipped">ارسال شده</SelectItem>
                          <SelectItem value="completed">تکمیل شده</SelectItem>
                          <SelectItem value="cancelled">لغو شده</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                    {order.items?.map((item) => (
                      <div key={item.id} className="flex items-center gap-3 text-sm py-1">
                        <span className="flex-1 line-clamp-1">{item.productName}</span>
                        <span className="text-muted-foreground">x{item.quantity}</span>
                        <span>{formatPrice(Number(item.price) * item.quantity)}</span>
                      </div>
                    ))}
                    <Separator className="my-2" />
                    <div className="flex items-center justify-between text-sm gap-2">
                      <span className="text-muted-foreground">{order.phone} | {order.shippingCity}</span>
                      <span className="font-bold text-primary">{formatPrice(order.total)}</span>
                    </div>
                  </Card>
                );
              })}
            </div>
          ) : (
            <Card className="p-8 text-center">
              <ShoppingCart className="w-12 h-12 mx-auto mb-3 text-muted-foreground" />
              <h3 className="font-bold">هنوز سفارشی ثبت نشده</h3>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="settings">
          <WooCommerceSettings />
        </TabsContent>
      </Tabs>
    </div>
  );
}

function CategoryManager({ categories }: { categories: Category[] }) {
  const { toast } = useToast();
  const sorted = [...categories].sort((a, b) => a.displayOrder - b.displayOrder);

  const updateCat = useMutation({
    mutationFn: ({ id, data }: { id: number; data: { visible?: boolean; displayOrder?: number; swapWithId?: number } }) =>
      apiRequest("PATCH", `/api/admin/categories/${id}`, data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/admin/categories"] });
      queryClient.invalidateQueries({ queryKey: ["/api/categories"] });
    },
  });

  const toggleVisible = (cat: Category) => {
    updateCat.mutate({ id: cat.id, data: { visible: !cat.visible } });
  };

  const moveUp = (cat: Category, index: number) => {
    if (index === 0) return;
    const prev = sorted[index - 1];
    updateCat.mutate({ id: cat.id, data: { swapWithId: prev.id } });
  };

  const moveDown = (cat: Category, index: number) => {
    if (index >= sorted.length - 1) return;
    const next = sorted[index + 1];
    updateCat.mutate({ id: cat.id, data: { swapWithId: next.id } });
  };

  if (categories.length === 0) {
    return (
      <Card className="p-8 text-center">
        <LayoutGrid className="w-12 h-12 mx-auto mb-3 text-muted-foreground" />
        <h3 className="font-bold mb-2">هنوز دسته‌بندی‌ای همگام‌سازی نشده</h3>
        <p className="text-sm text-muted-foreground">ابتدا با ووکامرس همگام‌سازی کنید تا دسته‌بندی‌ها نمایش داده شوند.</p>
      </Card>
    );
  }

  return (
    <div>
      <div className="flex items-center justify-between mb-4 gap-3 flex-wrap">
        <h2 className="font-bold" data-testid="text-admin-cats-list">مدیریت نمایش دسته‌بندی‌ها</h2>
        <p className="text-sm text-muted-foreground">ترتیب نمایش و نمایان بودن دسته‌بندی‌ها را تنظیم کنید.</p>
      </div>

      <Card className="overflow-hidden">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>تصویر</TableHead>
              <TableHead>نام</TableHead>
              <TableHead>نمایان</TableHead>
              <TableHead>ترتیب</TableHead>
              <TableHead>منبع</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {sorted.map((cat, index) => (
              <TableRow key={cat.id} data-testid={`row-admin-cat-${cat.id}`} className={!cat.visible ? "opacity-50" : ""}>
                <TableCell>
                  {cat.image ? (
                    <img src={cat.image} alt="" className="w-10 h-10 rounded-md object-cover" />
                  ) : (
                    <div className="w-10 h-10 rounded-md bg-muted flex items-center justify-center">
                      <LayoutGrid className="w-4 h-4 text-muted-foreground" />
                    </div>
                  )}
                </TableCell>
                <TableCell className="font-medium text-sm">{cat.name}</TableCell>
                <TableCell>
                  <Switch
                    checked={cat.visible}
                    onCheckedChange={() => toggleVisible(cat)}
                    data-testid={`switch-cat-visible-${cat.id}`}
                  />
                </TableCell>
                <TableCell>
                  <div className="flex items-center gap-1">
                    <Button
                      size="icon"
                      variant="ghost"
                      onClick={() => moveUp(cat, index)}
                      disabled={index === 0}
                      data-testid={`button-cat-up-${cat.id}`}
                    >
                      <ArrowUp className="w-4 h-4" />
                    </Button>
                    <Button
                      size="icon"
                      variant="ghost"
                      onClick={() => moveDown(cat, index)}
                      disabled={index >= sorted.length - 1}
                      data-testid={`button-cat-down-${cat.id}`}
                    >
                      <ArrowDown className="w-4 h-4" />
                    </Button>
                  </div>
                </TableCell>
                <TableCell>
                  {cat.wcId ? (
                    <Badge variant="outline" className="gap-1">
                      <Link2 className="w-3 h-3" />
                      WC #{cat.wcId}
                    </Badge>
                  ) : (
                    <Badge variant="secondary">محلی</Badge>
                  )}
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </Card>
    </div>
  );
}

function WooCommerceSettings() {
  const { toast } = useToast();
  const [form, setForm] = useState({
    wc_url: "",
    wc_key: "",
    wc_secret: "",
  });
  const [saving, setSaving] = useState(false);
  const [testing, setTesting] = useState(false);
  const [testResult, setTestResult] = useState<{ ok: boolean; message: string } | null>(null);

  const { data: settings } = useQuery<{ key: string; value: string | null }[]>({
    queryKey: ["/api/admin/settings"],
  });

  useEffect(() => {
    if (settings) {
      const url = settings.find((s) => s.key === "wc_url")?.value || "";
      const key = settings.find((s) => s.key === "wc_key")?.value || "";
      const secret = settings.find((s) => s.key === "wc_secret")?.value || "";
      setForm({ wc_url: url, wc_key: key, wc_secret: secret });
    }
  }, [settings]);

  const handleSave = async (e: React.FormEvent) => {
    e.preventDefault();
    setSaving(true);
    try {
      await apiRequest("POST", "/api/admin/settings", form);
      queryClient.invalidateQueries({ queryKey: ["/api/admin/settings"] });
      toast({ title: "تنظیمات ذخیره شد" });
    } catch (err: any) {
      toast({ title: "خطا", description: err.message, variant: "destructive" });
    } finally {
      setSaving(false);
    }
  };

  const handleTestConnection = async () => {
    setTesting(true);
    setTestResult(null);
    try {
      await apiRequest("POST", "/api/admin/settings", form);
      const res = await apiRequest("POST", "/api/admin/sync");
      const data = await res.json();
      setTestResult({ ok: true, message: data.message || "اتصال موفق!" });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/products"] });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/categories"] });
      queryClient.invalidateQueries({ queryKey: ["/api/admin/sync-status"] });
      queryClient.invalidateQueries({ queryKey: ["/api/products"] });
      queryClient.invalidateQueries({ queryKey: ["/api/categories"] });
    } catch (err: any) {
      setTestResult({ ok: false, message: err.message || "خطا در اتصال" });
    } finally {
      setTesting(false);
    }
  };

  return (
    <div className="max-w-xl space-y-4">
      <Card className="p-5">
        <div className="flex items-center gap-2 mb-4">
          <Globe className="w-5 h-5 text-primary" />
          <h3 className="font-bold" data-testid="text-wc-settings-title">تنظیمات اتصال به ووکامرس</h3>
        </div>
        <p className="text-sm text-muted-foreground mb-4">
          آدرس سایت وردپرسی و کلیدهای API ووکامرس خود را وارد کنید. برای دریافت کلیدها به مسیر
          وردپرس {'>'} ووکامرس {'>'} تنظیمات {'>'} پیشرفته {'>'} REST API مراجعه کنید.
        </p>
        <form onSubmit={handleSave} className="space-y-4">
          <div className="space-y-2">
            <Label>آدرس سایت وردپرسی</Label>
            <Input
              value={form.wc_url}
              onChange={(e) => setForm({ ...form, wc_url: e.target.value })}
              placeholder="https://your-wordpress-site.com"
              dir="ltr"
              data-testid="input-wc-url"
            />
          </div>
          <div className="space-y-2">
            <Label>Consumer Key</Label>
            <Input
              value={form.wc_key}
              onChange={(e) => setForm({ ...form, wc_key: e.target.value })}
              placeholder="ck_..."
              dir="ltr"
              data-testid="input-wc-key"
            />
          </div>
          <div className="space-y-2">
            <Label>Consumer Secret</Label>
            <Input
              value={form.wc_secret}
              onChange={(e) => setForm({ ...form, wc_secret: e.target.value })}
              placeholder="cs_..."
              dir="ltr"
              type="password"
              data-testid="input-wc-secret"
            />
          </div>

          <div className="flex items-center gap-2 flex-wrap">
            <Button type="submit" disabled={saving} className="gap-1" data-testid="button-save-wc-settings">
              <Save className="w-4 h-4" />
              {saving ? "ذخیره..." : "ذخیره تنظیمات"}
            </Button>
            <Button
              type="button"
              variant="outline"
              disabled={testing || !form.wc_url || !form.wc_key || !form.wc_secret}
              onClick={handleTestConnection}
              className="gap-1"
              data-testid="button-test-wc"
            >
              <RefreshCw className={`w-4 h-4 ${testing ? "animate-spin" : ""}`} />
              {testing ? "تست اتصال..." : "تست و همگام‌سازی"}
            </Button>
          </div>
        </form>

        {testResult && (
          <div className={`mt-4 p-3 rounded-md flex items-center gap-2 text-sm ${testResult.ok ? "bg-green-50 dark:bg-green-900/20 text-green-700 dark:text-green-400" : "bg-destructive/10 text-destructive"}`} data-testid="text-wc-test-result">
            {testResult.ok ? <CheckCircle className="w-4 h-4" /> : <AlertCircle className="w-4 h-4" />}
            {testResult.message}
          </div>
        )}
      </Card>

      <Card className="p-5">
        <h3 className="font-bold mb-3">راهنمای اتصال</h3>
        <ol className="space-y-2 text-sm text-muted-foreground list-decimal list-inside">
          <li>وارد پنل مدیریت وردپرس خود شوید</li>
          <li>به مسیر ووکامرس {'>'} تنظیمات {'>'} پیشرفته {'>'} REST API بروید</li>
          <li>یک کلید جدید با دسترسی "خواندن" ایجاد کنید</li>
          <li>Consumer Key و Consumer Secret را کپی و در فیلدهای بالا وارد کنید</li>
          <li>دکمه "تست و همگام‌سازی" را بزنید</li>
        </ol>
      </Card>
    </div>
  );
}
