import { Link } from "wouter";
import { Trash2, Minus, Plus, ShoppingCart, ArrowLeft } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Separator } from "@/components/ui/separator";
import { useCart } from "@/lib/cart";
import { formatPrice } from "@/lib/format";
import { useAuth } from "@/lib/auth";

export default function CartPage() {
  const { items, removeItem, updateQuantity, totalPrice, totalItems } = useCart();
  const { user } = useAuth();

  if (items.length === 0) {
    return (
      <div className="max-w-7xl mx-auto px-4 py-16 text-center">
        <div className="max-w-md mx-auto">
          <ShoppingCart className="w-16 h-16 mx-auto text-muted-foreground mb-4" />
          <h1 className="text-xl font-bold mb-2" data-testid="text-empty-cart">سبد خرید شما خالی است</h1>
          <p className="text-muted-foreground mb-6" data-testid="text-empty-cart-desc">
            محصولات مورد نظر خود را به سبد خرید اضافه کنید
          </p>
          <Link href="/products">
            <Button className="gap-2" data-testid="link-continue-shopping">
              مشاهده محصولات
              <ArrowLeft className="w-4 h-4" />
            </Button>
          </Link>
        </div>
      </div>
    );
  }

  return (
    <div className="max-w-7xl mx-auto px-4 py-6">
      <h1 className="text-2xl font-bold mb-6" data-testid="text-cart-title">سبد خرید</h1>

      <div className="grid md:grid-cols-3 gap-6">
        <div className="md:col-span-2 space-y-3">
          {items.map((item) => (
            <Card key={item.productId} className="p-4" data-testid={`card-cart-item-${item.productId}`}>
              <div className="flex gap-4">
                <Link href={`/products/${item.product.slug}`}>
                  <div className="w-20 h-20 rounded-md overflow-hidden shrink-0 cursor-pointer">
                    <img
                      src={item.product.images?.[0] || "/images/placeholder.png"}
                      alt={item.product.name}
                      className="w-full h-full object-cover"
                    />
                  </div>
                </Link>
                <div className="flex-1 min-w-0">
                  <Link href={`/products/${item.product.slug}`}>
                    <h3 className="font-medium text-sm line-clamp-1 cursor-pointer" data-testid={`text-cart-name-${item.productId}`}>
                      {item.product.name}
                    </h3>
                  </Link>
                  <p className="text-sm text-primary font-bold mt-1" data-testid={`text-cart-price-${item.productId}`}>
                    {formatPrice(item.product.salePrice || item.product.price)}
                  </p>

                  <div className="flex items-center justify-between mt-3 flex-wrap gap-2">
                    <div className="flex items-center border rounded-md">
                      <Button
                        size="icon"
                        variant="ghost"
                        className="h-8 w-8"
                        onClick={() => updateQuantity(item.productId, item.quantity - 1)}
                        data-testid={`button-cart-minus-${item.productId}`}
                      >
                        <Minus className="w-3 h-3" />
                      </Button>
                      <span className="w-8 text-center text-sm font-medium">{item.quantity}</span>
                      <Button
                        size="icon"
                        variant="ghost"
                        className="h-8 w-8"
                        onClick={() => updateQuantity(item.productId, item.quantity + 1)}
                        data-testid={`button-cart-plus-${item.productId}`}
                      >
                        <Plus className="w-3 h-3" />
                      </Button>
                    </div>

                    <Button
                      size="icon"
                      variant="ghost"
                      onClick={() => removeItem(item.productId)}
                      data-testid={`button-cart-remove-${item.productId}`}
                    >
                      <Trash2 className="w-4 h-4 text-destructive" />
                    </Button>
                  </div>
                </div>
              </div>
            </Card>
          ))}
        </div>

        <div>
          <Card className="p-5 sticky top-20">
            <h3 className="font-bold mb-4" data-testid="text-order-summary">خلاصه سفارش</h3>
            <div className="space-y-3 text-sm">
              <div className="flex items-center justify-between gap-2">
                <span className="text-muted-foreground">تعداد کالا:</span>
                <span data-testid="text-total-items">{totalItems} عدد</span>
              </div>
              <Separator />
              <div className="flex items-center justify-between gap-2">
                <span className="font-bold">مبلغ قابل پرداخت:</span>
                <span className="text-primary font-bold" data-testid="text-total-price">{formatPrice(totalPrice)}</span>
              </div>
            </div>

            <Link href={user ? "/checkout" : "/login?redirect=/checkout"}>
              <Button className="w-full mt-5 gap-2" data-testid="button-checkout">
                {user ? "ادامه خرید" : "ورود و ادامه"}
                <ArrowLeft className="w-4 h-4" />
              </Button>
            </Link>
          </Card>
        </div>
      </div>
    </div>
  );
}
