import { useState } from "react";
import { useLocation } from "wouter";
import { ArrowLeft, CheckCircle } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Separator } from "@/components/ui/separator";
import { useCart } from "@/lib/cart";
import { useAuth } from "@/lib/auth";
import { formatPrice } from "@/lib/format";
import { apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

export default function Checkout() {
  const { items, totalPrice, clearCart } = useCart();
  const { user } = useAuth();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [loading, setLoading] = useState(false);
  const [success, setSuccess] = useState(false);
  const [orderId, setOrderId] = useState<number | null>(null);

  const [form, setForm] = useState({
    name: user?.name || "",
    phone: user?.phone || "",
    address: user?.address || "",
    city: user?.city || "",
    postalCode: user?.postalCode || "",
    notes: "",
  });

  if (!user) {
    navigate("/login?redirect=/checkout");
    return null;
  }

  if (items.length === 0 && !success) {
    navigate("/cart");
    return null;
  }

  if (success) {
    return (
      <div className="max-w-md mx-auto px-4 py-16 text-center">
        <div className="w-16 h-16 rounded-full bg-green-100 dark:bg-green-900/30 mx-auto mb-4 flex items-center justify-center">
          <CheckCircle className="w-8 h-8 text-green-600 dark:text-green-400" />
        </div>
        <h1 className="text-xl font-bold mb-2" data-testid="text-order-success">سفارش شما با موفقیت ثبت شد</h1>
        <p className="text-muted-foreground mb-6" data-testid="text-order-id">شماره سفارش: {orderId}</p>
        <div className="flex flex-col gap-3">
          <Button onClick={() => navigate(`/account`)} data-testid="button-view-orders">مشاهده سفارشات</Button>
          <Button variant="outline" onClick={() => navigate("/products")} data-testid="button-continue">ادامه خرید</Button>
        </div>
      </div>
    );
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    try {
      const res = await apiRequest("POST", "/api/orders", {
        shippingAddress: form.address,
        shippingCity: form.city,
        shippingPostalCode: form.postalCode,
        phone: form.phone,
        notes: form.notes,
        items: items.map((i) => ({
          productId: i.productId,
          quantity: i.quantity,
        })),
      });
      const data = await res.json();
      setOrderId(data.id);
      setSuccess(true);
      clearCart();
    } catch (err: any) {
      toast({
        title: "خطا",
        description: err.message || "خطایی رخ داد",
        variant: "destructive",
      });
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="max-w-4xl mx-auto px-4 py-6">
      <h1 className="text-2xl font-bold mb-6" data-testid="text-checkout-title">تکمیل سفارش</h1>

      <form onSubmit={handleSubmit}>
        <div className="grid md:grid-cols-3 gap-6">
          <div className="md:col-span-2 space-y-4">
            <Card className="p-5 space-y-4">
              <h3 className="font-bold" data-testid="text-shipping-info">اطلاعات ارسال</h3>
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="name">نام و نام خانوادگی</Label>
                  <Input
                    id="name"
                    value={form.name}
                    onChange={(e) => setForm({ ...form, name: e.target.value })}
                    required
                    data-testid="input-name"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="phone">شماره تلفن</Label>
                  <Input
                    id="phone"
                    value={form.phone}
                    onChange={(e) => setForm({ ...form, phone: e.target.value })}
                    required
                    dir="ltr"
                    data-testid="input-phone"
                  />
                </div>
              </div>
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="city">شهر</Label>
                  <Input
                    id="city"
                    value={form.city}
                    onChange={(e) => setForm({ ...form, city: e.target.value })}
                    required
                    data-testid="input-city"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="postalCode">کد پستی</Label>
                  <Input
                    id="postalCode"
                    value={form.postalCode}
                    onChange={(e) => setForm({ ...form, postalCode: e.target.value })}
                    dir="ltr"
                    data-testid="input-postal-code"
                  />
                </div>
              </div>
              <div className="space-y-2">
                <Label htmlFor="address">آدرس کامل</Label>
                <Textarea
                  id="address"
                  value={form.address}
                  onChange={(e) => setForm({ ...form, address: e.target.value })}
                  required
                  className="resize-none"
                  rows={3}
                  data-testid="input-address"
                />
              </div>
              <div className="space-y-2">
                <Label htmlFor="notes">توضیحات (اختیاری)</Label>
                <Textarea
                  id="notes"
                  value={form.notes}
                  onChange={(e) => setForm({ ...form, notes: e.target.value })}
                  className="resize-none"
                  rows={2}
                  data-testid="input-notes"
                />
              </div>
            </Card>
          </div>

          <div>
            <Card className="p-5 sticky top-20">
              <h3 className="font-bold mb-4" data-testid="text-checkout-summary">خلاصه سفارش</h3>
              <div className="space-y-3 text-sm">
                {items.map((item) => (
                  <div key={item.productId} className="flex items-center justify-between gap-2">
                    <span className="text-muted-foreground line-clamp-1">{item.product.name} x {item.quantity}</span>
                    <span className="shrink-0">{formatPrice(Number(item.product.salePrice || item.product.price) * item.quantity)}</span>
                  </div>
                ))}
                <Separator />
                <div className="flex items-center justify-between gap-2 font-bold">
                  <span>جمع کل:</span>
                  <span className="text-primary" data-testid="text-checkout-total">{formatPrice(totalPrice)}</span>
                </div>
              </div>

              <Button type="submit" className="w-full mt-5 gap-2" disabled={loading} data-testid="button-place-order">
                {loading ? "در حال ثبت..." : "ثبت سفارش"}
                <ArrowLeft className="w-4 h-4" />
              </Button>
            </Card>
          </div>
        </div>
      </form>
    </div>
  );
}
