import { useQuery } from "@tanstack/react-query";
import { Link } from "wouter";
import { ArrowLeft, Truck, Shield, Headphones, RotateCcw } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Skeleton } from "@/components/ui/skeleton";
import ProductCard from "@/components/product-card";
import type { Product, Category } from "@shared/schema";

export default function Home() {
  const { data: featured, isLoading: loadingFeatured } = useQuery<Product[]>({
    queryKey: ["/api/products?featured=true"],
  });

  const { data: categories, isLoading: loadingCats } = useQuery<Category[]>({
    queryKey: ["/api/categories"],
  });

  const { data: latest, isLoading: loadingLatest } = useQuery<Product[]>({
    queryKey: ["/api/products?limit=8"],
  });

  const features = [
    { icon: Truck, title: "ارسال رایگان", desc: "ارسال رایگان برای سفارش‌های بالای ۵۰۰ هزار تومان" },
    { icon: Shield, title: "ضمانت اصالت", desc: "تضمین اصالت تمام محصولات" },
    { icon: Headphones, title: "پشتیبانی ۲۴/۷", desc: "پشتیبانی آنلاین در تمام ساعات" },
    { icon: RotateCcw, title: "بازگشت کالا", desc: "امکان بازگشت تا ۷ روز" },
  ];

  return (
    <div className="min-h-screen">
      <section className="relative bg-gradient-to-l from-primary/10 to-primary/5 dark:from-primary/5 dark:to-transparent">
        <div className="max-w-7xl mx-auto px-4 py-16 md:py-24">
          <div className="max-w-lg">
            <h1 className="text-3xl md:text-5xl font-bold leading-tight mb-4" data-testid="text-hero-title">
              بهترین محصولات، بهترین قیمت‌ها
            </h1>
            <p className="text-muted-foreground text-base md:text-lg mb-8 leading-relaxed" data-testid="text-hero-desc">
              از میان هزاران محصول با کیفیت، بهترین‌ها را انتخاب کنید و با خیال راحت خرید کنید.
            </p>
            <Link href="/products">
              <Button size="lg" className="gap-2" data-testid="button-hero-cta">
                مشاهده محصولات
                <ArrowLeft className="w-4 h-4" />
              </Button>
            </Link>
          </div>
        </div>
      </section>

      <section className="max-w-7xl mx-auto px-4 py-8">
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
          {features.map((f, i) => (
            <Card key={i} className="p-4 flex flex-col items-center text-center gap-2" data-testid={`card-feature-${i}`}>
              <div className="w-12 h-12 rounded-full bg-primary/10 flex items-center justify-center mb-1">
                <f.icon className="w-5 h-5 text-primary" />
              </div>
              <h3 className="font-semibold text-sm">{f.title}</h3>
              <p className="text-xs text-muted-foreground">{f.desc}</p>
            </Card>
          ))}
        </div>
      </section>

      {categories && categories.length > 0 && (
        <section className="max-w-7xl mx-auto px-4 py-8">
          <div className="flex items-center justify-between mb-6">
            <h2 className="text-xl font-bold" data-testid="text-categories-title">دسته‌بندی‌ها</h2>
            <Link href="/products">
              <Button variant="ghost" size="sm" className="gap-1" data-testid="link-all-categories">
                مشاهده همه
                <ArrowLeft className="w-4 h-4" />
              </Button>
            </Link>
          </div>
          <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4">
            {loadingCats
              ? Array.from({ length: 6 }).map((_, i) => (
                  <Skeleton key={i} className="aspect-square rounded-md" />
                ))
              : categories.map((cat) => (
                  <Link key={cat.id} href={`/products?category=${cat.slug}`}>
                    <Card className="group overflow-visible hover-elevate cursor-pointer" data-testid={`card-category-${cat.id}`}>
                      <div className="aspect-square overflow-hidden rounded-t-md">
                        <img
                          src={cat.image || "/images/placeholder.png"}
                          alt={cat.name}
                          className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-105"
                        />
                      </div>
                      <div className="p-3 text-center">
                        <h3 className="text-sm font-medium">{cat.name}</h3>
                      </div>
                    </Card>
                  </Link>
                ))}
          </div>
        </section>
      )}

      {featured && featured.length > 0 && (
        <section className="max-w-7xl mx-auto px-4 py-8">
          <div className="flex items-center justify-between mb-6">
            <h2 className="text-xl font-bold" data-testid="text-featured-title">محصولات ویژه</h2>
            <Link href="/products?featured=true">
              <Button variant="ghost" size="sm" className="gap-1" data-testid="link-all-featured">
                مشاهده همه
                <ArrowLeft className="w-4 h-4" />
              </Button>
            </Link>
          </div>
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
            {loadingFeatured
              ? Array.from({ length: 4 }).map((_, i) => (
                  <Skeleton key={i} className="aspect-[3/4] rounded-md" />
                ))
              : featured.slice(0, 4).map((p) => <ProductCard key={p.id} product={p} />)}
          </div>
        </section>
      )}

      <section className="max-w-7xl mx-auto px-4 py-8">
        <div className="flex items-center justify-between mb-6">
          <h2 className="text-xl font-bold" data-testid="text-latest-title">جدیدترین محصولات</h2>
          <Link href="/products">
            <Button variant="ghost" size="sm" className="gap-1" data-testid="link-all-latest">
              مشاهده همه
              <ArrowLeft className="w-4 h-4" />
            </Button>
          </Link>
        </div>
        <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
          {loadingLatest
            ? Array.from({ length: 8 }).map((_, i) => (
                <Skeleton key={i} className="aspect-[3/4] rounded-md" />
              ))
            : latest?.map((p) => <ProductCard key={p.id} product={p} />)}
        </div>
      </section>
    </div>
  );
}
