import { useState } from "react";
import { useLocation, useSearch } from "wouter";
import { Phone, ArrowLeft, KeyRound } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { InputOTP, InputOTPGroup, InputOTPSlot } from "@/components/ui/input-otp";
import { useAuth } from "@/lib/auth";
import { useToast } from "@/hooks/use-toast";

export default function Login() {
  const [step, setStep] = useState<"phone" | "otp">("phone");
  const [phone, setPhone] = useState("");
  const [code, setCode] = useState("");
  const [loading, setLoading] = useState(false);
  const { login, verifyOtp } = useAuth();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const searchParams = new URLSearchParams(useSearch());
  const redirect = searchParams.get("redirect") || "/account";

  const handleSendOtp = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!phone || phone.length < 10) {
      toast({ title: "خطا", description: "شماره تلفن معتبر وارد کنید", variant: "destructive" });
      return;
    }
    setLoading(true);
    try {
      await login(phone);
      setStep("otp");
      toast({ title: "کد تایید ارسال شد", description: "کد ۶ رقمی ارسال شده را وارد کنید" });
    } catch (err: any) {
      toast({ title: "خطا", description: err.message || "خطایی رخ داد", variant: "destructive" });
    } finally {
      setLoading(false);
    }
  };

  const handleVerifyOtp = async (e: React.FormEvent) => {
    e.preventDefault();
    if (code.length !== 6) {
      toast({ title: "خطا", description: "کد ۶ رقمی را کامل وارد کنید", variant: "destructive" });
      return;
    }
    setLoading(true);
    try {
      const success = await verifyOtp(phone, code);
      if (success) {
        toast({ title: "خوش آمدید!" });
        navigate(redirect);
      } else {
        toast({ title: "خطا", description: "کد وارد شده اشتباه است", variant: "destructive" });
      }
    } catch (err: any) {
      toast({ title: "خطا", description: err.message || "خطایی رخ داد", variant: "destructive" });
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-[60vh] flex items-center justify-center px-4 py-12">
      <Card className="w-full max-w-md p-6 space-y-6">
        <div className="text-center">
          <div className="w-14 h-14 rounded-full bg-primary/10 mx-auto mb-4 flex items-center justify-center">
            {step === "phone" ? (
              <Phone className="w-6 h-6 text-primary" />
            ) : (
              <KeyRound className="w-6 h-6 text-primary" />
            )}
          </div>
          <h1 className="text-xl font-bold" data-testid="text-login-title">
            {step === "phone" ? "ورود / ثبت‌نام" : "کد تایید"}
          </h1>
          <p className="text-sm text-muted-foreground mt-1" data-testid="text-login-desc">
            {step === "phone"
              ? "شماره موبایل خود را وارد کنید"
              : `کد ارسال شده به ${phone} را وارد کنید`}
          </p>
        </div>

        {step === "phone" ? (
          <form onSubmit={handleSendOtp} className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="phone">شماره موبایل</Label>
              <Input
                id="phone"
                type="tel"
                placeholder="09123456789"
                value={phone}
                onChange={(e) => setPhone(e.target.value)}
                dir="ltr"
                className="text-center text-lg tracking-wider"
                data-testid="input-login-phone"
              />
            </div>
            <Button type="submit" className="w-full gap-2" disabled={loading} data-testid="button-send-otp">
              {loading ? "در حال ارسال..." : "ارسال کد تایید"}
              <ArrowLeft className="w-4 h-4" />
            </Button>
          </form>
        ) : (
          <form onSubmit={handleVerifyOtp} className="space-y-4">
            <div className="flex justify-center" dir="ltr">
              <InputOTP maxLength={6} value={code} onChange={setCode} data-testid="input-otp">
                <InputOTPGroup>
                  <InputOTPSlot index={0} />
                  <InputOTPSlot index={1} />
                  <InputOTPSlot index={2} />
                  <InputOTPSlot index={3} />
                  <InputOTPSlot index={4} />
                  <InputOTPSlot index={5} />
                </InputOTPGroup>
              </InputOTP>
            </div>
            <Button type="submit" className="w-full gap-2" disabled={loading} data-testid="button-verify-otp">
              {loading ? "در حال بررسی..." : "تایید و ورود"}
              <ArrowLeft className="w-4 h-4" />
            </Button>
            <Button
              type="button"
              variant="ghost"
              className="w-full"
              onClick={() => { setStep("phone"); setCode(""); }}
              data-testid="button-change-phone"
            >
              تغییر شماره موبایل
            </Button>
          </form>
        )}
      </Card>
    </div>
  );
}
