import { useQuery } from "@tanstack/react-query";
import { useParams, Link } from "wouter";
import { ShoppingCart, Minus, Plus, ArrowRight, Check, Package } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Skeleton } from "@/components/ui/skeleton";
import { Separator } from "@/components/ui/separator";
import { useCart } from "@/lib/cart";
import { formatPrice } from "@/lib/format";
import { useToast } from "@/hooks/use-toast";
import type { Product } from "@shared/schema";
import { useState } from "react";
import ProductCard from "@/components/product-card";

export default function ProductDetail() {
  const { slug } = useParams<{ slug: string }>();
  const { addItem } = useCart();
  const { toast } = useToast();
  const [quantity, setQuantity] = useState(1);
  const [selectedImage, setSelectedImage] = useState(0);

  const { data: product, isLoading } = useQuery<Product>({
    queryKey: ["/api/products/" + slug],
  });

  const { data: related } = useQuery<Product[]>({
    queryKey: [`/api/products?category=${product?.categoryId}&exclude=${product?.id}`],
    enabled: !!product?.categoryId,
  });

  if (isLoading) {
    return (
      <div className="max-w-7xl mx-auto px-4 py-6">
        <div className="grid md:grid-cols-2 gap-8">
          <Skeleton className="aspect-square rounded-md" />
          <div className="space-y-4">
            <Skeleton className="h-8 w-3/4" />
            <Skeleton className="h-6 w-1/2" />
            <Skeleton className="h-24 w-full" />
          </div>
        </div>
      </div>
    );
  }

  if (!product) {
    return (
      <div className="max-w-7xl mx-auto px-4 py-12 text-center">
        <h1 className="text-xl font-bold mb-4" data-testid="text-product-not-found">محصول یافت نشد</h1>
        <Link href="/products">
          <Button data-testid="link-back-products">بازگشت به محصولات</Button>
        </Link>
      </div>
    );
  }

  const discount = product.salePrice
    ? Math.round(((Number(product.price) - Number(product.salePrice)) / Number(product.price)) * 100)
    : 0;

  const images = product.images?.length ? product.images : ["/images/placeholder.png"];

  const handleAddToCart = () => {
    addItem(product, quantity);
    toast({
      title: "به سبد خرید اضافه شد",
      description: `${product.name} x ${quantity}`,
    });
  };

  return (
    <div className="max-w-7xl mx-auto px-4 py-6">
      <div className="flex items-center gap-2 text-sm text-muted-foreground mb-6 flex-wrap">
        <Link href="/">
          <span className="cursor-pointer hover:text-foreground" data-testid="breadcrumb-home">خانه</span>
        </Link>
        <ArrowRight className="w-3 h-3" />
        <Link href="/products">
          <span className="cursor-pointer hover:text-foreground" data-testid="breadcrumb-products">محصولات</span>
        </Link>
        <ArrowRight className="w-3 h-3" />
        <span className="text-foreground" data-testid="breadcrumb-current">{product.name}</span>
      </div>

      <div className="grid md:grid-cols-2 gap-8">
        <div className="space-y-3">
          <div className="relative aspect-square overflow-hidden rounded-md bg-card">
            <img
              src={images[selectedImage]}
              alt={product.name}
              className="w-full h-full object-cover"
              data-testid="img-product-main"
            />
            {discount > 0 && (
              <Badge className="absolute top-3 right-3 bg-destructive text-destructive-foreground text-sm" data-testid="badge-product-discount">
                {discount}%-
              </Badge>
            )}
          </div>
          {images.length > 1 && (
            <div className="flex gap-2 overflow-x-auto">
              {images.map((img, i) => (
                <button
                  key={i}
                  className={`w-16 h-16 rounded-md overflow-hidden border-2 shrink-0 ${
                    selectedImage === i ? "border-primary" : "border-transparent"
                  }`}
                  onClick={() => setSelectedImage(i)}
                  data-testid={`button-thumb-${i}`}
                >
                  <img src={img} alt="" className="w-full h-full object-cover" />
                </button>
              ))}
            </div>
          )}
        </div>

        <div className="space-y-6">
          <div>
            <h1 className="text-2xl font-bold mb-2" data-testid="text-product-title">{product.name}</h1>
            {product.sku && (
              <p className="text-sm text-muted-foreground" data-testid="text-product-sku">کد محصول: {product.sku}</p>
            )}
          </div>

          <div className="flex items-center gap-4">
            {product.salePrice ? (
              <>
                <span className="text-2xl font-bold text-primary" data-testid="text-detail-sale-price">
                  {formatPrice(product.salePrice)}
                </span>
                <span className="text-lg text-muted-foreground line-through" data-testid="text-detail-original-price">
                  {formatPrice(product.price)}
                </span>
              </>
            ) : (
              <span className="text-2xl font-bold" data-testid="text-detail-price">
                {formatPrice(product.price)}
              </span>
            )}
          </div>

          <Separator />

          <div className="flex items-center gap-3">
            {product.inStock ? (
              <div className="flex items-center gap-2 text-sm text-green-600 dark:text-green-400">
                <Check className="w-4 h-4" />
                <span data-testid="text-in-stock">موجود در انبار ({product.stockQuantity} عدد)</span>
              </div>
            ) : (
              <div className="flex items-center gap-2 text-sm text-destructive">
                <Package className="w-4 h-4" />
                <span data-testid="text-out-of-stock">ناموجود</span>
              </div>
            )}
          </div>

          {product.shortDescription && (
            <p className="text-sm text-muted-foreground leading-relaxed" data-testid="text-short-desc">
              {product.shortDescription}
            </p>
          )}

          {product.inStock && (
            <div className="flex items-center gap-4 flex-wrap">
              <div className="flex items-center border rounded-md">
                <Button
                  size="icon"
                  variant="ghost"
                  onClick={() => setQuantity(Math.max(1, quantity - 1))}
                  data-testid="button-qty-minus"
                >
                  <Minus className="w-4 h-4" />
                </Button>
                <span className="w-10 text-center font-medium" data-testid="text-quantity">{quantity}</span>
                <Button
                  size="icon"
                  variant="ghost"
                  onClick={() => setQuantity(Math.min(product.stockQuantity, quantity + 1))}
                  data-testid="button-qty-plus"
                >
                  <Plus className="w-4 h-4" />
                </Button>
              </div>

              <Button className="gap-2 flex-1 md:flex-none" onClick={handleAddToCart} data-testid="button-add-to-cart">
                <ShoppingCart className="w-4 h-4" />
                افزودن به سبد خرید
              </Button>
            </div>
          )}
        </div>
      </div>

      <div className="mt-10">
        <Tabs defaultValue="description" dir="rtl">
          <TabsList>
            <TabsTrigger value="description" data-testid="tab-description">توضیحات</TabsTrigger>
            <TabsTrigger value="specs" data-testid="tab-specs">مشخصات</TabsTrigger>
          </TabsList>
          <TabsContent value="description" className="mt-4">
            <Card className="p-6">
              <div className="prose dark:prose-invert max-w-none text-sm leading-relaxed" data-testid="text-description">
                {product.description || "توضیحاتی برای این محصول ثبت نشده است."}
              </div>
            </Card>
          </TabsContent>
          <TabsContent value="specs" className="mt-4">
            <Card className="p-6">
              {product.attributes && typeof product.attributes === "object" ? (
                <div className="space-y-2">
                  {Object.entries(product.attributes as Record<string, string>).map(([key, val]) => (
                    <div key={key} className="flex items-center gap-2 text-sm py-2 border-b last:border-0">
                      <span className="font-medium w-32 shrink-0">{key}:</span>
                      <span className="text-muted-foreground">{val}</span>
                    </div>
                  ))}
                </div>
              ) : (
                <p className="text-sm text-muted-foreground" data-testid="text-no-specs">مشخصاتی ثبت نشده است.</p>
              )}
            </Card>
          </TabsContent>
        </Tabs>
      </div>

      {related && related.length > 0 && (
        <section className="mt-10">
          <h2 className="text-xl font-bold mb-6" data-testid="text-related-title">محصولات مرتبط</h2>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
            {related.slice(0, 4).map((p) => (
              <ProductCard key={p.id} product={p} />
            ))}
          </div>
        </section>
      )}
    </div>
  );
}
