import { useQuery } from "@tanstack/react-query";
import { useLocation, useSearch } from "wouter";
import { SlidersHorizontal, LayoutGrid, List, X } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Skeleton } from "@/components/ui/skeleton";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import ProductCard from "@/components/product-card";
import type { Product, Category } from "@shared/schema";
import { useState, useMemo } from "react";

export default function Products() {
  const searchParams = new URLSearchParams(useSearch());
  const categorySlug = searchParams.get("category") || "";
  const searchQuery = searchParams.get("search") || "";
  const isFeatured = searchParams.get("featured") === "true";
  const [sortBy, setSortBy] = useState("newest");
  const [priceRange, setPriceRange] = useState<[string, string]>(["", ""]);
  const [filterOpen, setFilterOpen] = useState(false);

  const { data: products, isLoading } = useQuery<Product[]>({
    queryKey: ["/api/products"],
  });

  const { data: categories } = useQuery<Category[]>({
    queryKey: ["/api/categories"],
  });

  const [, navigate] = useLocation();

  const filteredProducts = useMemo(() => {
    if (!products) return [];
    let filtered = [...products];

    if (categorySlug) {
      const cat = categories?.find((c) => c.slug === categorySlug);
      if (cat) {
        filtered = filtered.filter((p) => p.categoryId === cat.id);
      }
    }

    if (searchQuery) {
      const q = searchQuery.toLowerCase();
      filtered = filtered.filter(
        (p) => p.name.toLowerCase().includes(q) || p.description?.toLowerCase().includes(q)
      );
    }

    if (isFeatured) {
      filtered = filtered.filter((p) => p.featured);
    }

    const minPrice = priceRange[0] ? Number(priceRange[0]) : 0;
    const maxPrice = priceRange[1] ? Number(priceRange[1]) : Infinity;
    filtered = filtered.filter((p) => {
      const price = Number(p.salePrice || p.price);
      return price >= minPrice && price <= maxPrice;
    });

    switch (sortBy) {
      case "price-asc":
        filtered.sort((a, b) => Number(a.salePrice || a.price) - Number(b.salePrice || b.price));
        break;
      case "price-desc":
        filtered.sort((a, b) => Number(b.salePrice || b.price) - Number(a.salePrice || a.price));
        break;
      case "name":
        filtered.sort((a, b) => a.name.localeCompare(b.name, "fa"));
        break;
      default:
        filtered.sort((a, b) => b.id - a.id);
    }

    return filtered;
  }, [products, categorySlug, searchQuery, isFeatured, sortBy, priceRange, categories]);

  const currentCategory = categories?.find((c) => c.slug === categorySlug);

  const FilterSidebar = () => (
    <div className="space-y-6">
      <div>
        <h3 className="font-semibold mb-3" data-testid="text-filter-categories">دسته‌بندی‌ها</h3>
        <div className="space-y-1">
          <div
            className={`p-2 rounded-md cursor-pointer text-sm ${!categorySlug ? "bg-primary/10 text-primary font-medium" : "hover-elevate"}`}
            onClick={() => navigate("/products")}
            data-testid="filter-all"
          >
            همه محصولات
          </div>
          {categories?.map((cat) => (
            <div
              key={cat.id}
              className={`p-2 rounded-md cursor-pointer text-sm ${categorySlug === cat.slug ? "bg-primary/10 text-primary font-medium" : "hover-elevate"}`}
              onClick={() => navigate(`/products?category=${cat.slug}`)}
              data-testid={`filter-cat-${cat.id}`}
            >
              {cat.name}
            </div>
          ))}
        </div>
      </div>

      <div>
        <h3 className="font-semibold mb-3" data-testid="text-filter-price">محدوده قیمت</h3>
        <div className="flex items-center gap-2">
          <Input
            type="number"
            placeholder="از"
            value={priceRange[0]}
            onChange={(e) => setPriceRange([e.target.value, priceRange[1]])}
            className="text-sm"
            data-testid="input-price-min"
          />
          <span className="text-muted-foreground">تا</span>
          <Input
            type="number"
            placeholder="تا"
            value={priceRange[1]}
            onChange={(e) => setPriceRange([priceRange[0], e.target.value])}
            className="text-sm"
            data-testid="input-price-max"
          />
        </div>
      </div>

      {(priceRange[0] || priceRange[1] || categorySlug) && (
        <Button
          variant="outline"
          size="sm"
          className="w-full gap-1"
          onClick={() => {
            setPriceRange(["", ""]);
            navigate("/products");
          }}
          data-testid="button-clear-filters"
        >
          <X className="w-4 h-4" />
          حذف فیلترها
        </Button>
      )}
    </div>
  );

  return (
    <div className="max-w-7xl mx-auto px-4 py-6">
      <div className="flex items-center justify-between mb-6 flex-wrap gap-3">
        <div>
          <h1 className="text-2xl font-bold" data-testid="text-products-title">
            {currentCategory ? currentCategory.name : searchQuery ? `نتایج جستجو: ${searchQuery}` : isFeatured ? "محصولات ویژه" : "همه محصولات"}
          </h1>
          <p className="text-sm text-muted-foreground mt-1" data-testid="text-products-count">
            {filteredProducts.length} محصول
          </p>
        </div>

        <div className="flex items-center gap-2">
          <Sheet open={filterOpen} onOpenChange={setFilterOpen}>
            <SheetTrigger asChild>
              <Button variant="outline" size="sm" className="md:hidden gap-1" data-testid="button-mobile-filter">
                <SlidersHorizontal className="w-4 h-4" />
                فیلتر
              </Button>
            </SheetTrigger>
            <SheetContent side="right" className="w-80">
              <div className="mt-6">
                <FilterSidebar />
              </div>
            </SheetContent>
          </Sheet>

          <Select value={sortBy} onValueChange={setSortBy}>
            <SelectTrigger className="w-40 text-sm" data-testid="select-sort">
              <SelectValue placeholder="مرتب‌سازی" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="newest">جدیدترین</SelectItem>
              <SelectItem value="price-asc">ارزان‌ترین</SelectItem>
              <SelectItem value="price-desc">گران‌ترین</SelectItem>
              <SelectItem value="name">نام</SelectItem>
            </SelectContent>
          </Select>
        </div>
      </div>

      <div className="flex gap-6">
        <aside className="hidden md:block w-60 shrink-0">
          <Card className="p-4 sticky top-20">
            <FilterSidebar />
          </Card>
        </aside>

        <div className="flex-1">
          {isLoading ? (
            <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
              {Array.from({ length: 9 }).map((_, i) => (
                <Skeleton key={i} className="aspect-[3/4] rounded-md" />
              ))}
            </div>
          ) : filteredProducts.length === 0 ? (
            <Card className="p-12 text-center">
              <p className="text-muted-foreground" data-testid="text-no-products">محصولی یافت نشد</p>
            </Card>
          ) : (
            <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
              {filteredProducts.map((p) => (
                <ProductCard key={p.id} product={p} />
              ))}
            </div>
          )}
        </div>
      </div>
    </div>
  );
}
