import type { Express } from "express";
import { createServer, type Server } from "http";
import session from "express-session";
import pgSession from "connect-pg-simple";
import { storage } from "./storage";
import { loginSchema, verifyOtpSchema } from "@shared/schema";
import { log } from "./index";

declare module "express-session" {
  interface SessionData {
    userId: number;
  }
}

export async function registerRoutes(
  httpServer: Server,
  app: Express
): Promise<Server> {
  const PgStore = pgSession(session);

  app.use(
    session({
      store: new PgStore({
        conString: process.env.DATABASE_URL,
        createTableIfMissing: true,
      }),
      secret: process.env.SESSION_SECRET || "shop-secret-key-change-me",
      resave: false,
      saveUninitialized: false,
      cookie: {
        maxAge: 30 * 24 * 60 * 60 * 1000,
        httpOnly: true,
        sameSite: "lax",
      },
    })
  );

  function requireAuth(req: any, res: any, next: any) {
    if (!req.session.userId) {
      return res.status(401).json({ message: "لطفاً وارد شوید" });
    }
    next();
  }

  async function requireAdmin(req: any, res: any, next: any) {
    if (!req.session.userId) {
      return res.status(401).json({ message: "لطفاً وارد شوید" });
    }
    const user = await storage.getUser(req.session.userId);
    if (!user || user.role !== "admin") {
      return res.status(403).json({ message: "دسترسی غیرمجاز" });
    }
    next();
  }

  app.post("/api/auth/send-otp", async (req, res) => {
    try {
      const { phone } = loginSchema.parse(req.body);
      const code = Math.floor(100000 + Math.random() * 900000).toString();
      const expiresAt = new Date(Date.now() + 5 * 60 * 1000);
      await storage.createOtp(phone, code, expiresAt);
      log(`OTP for ${phone}: ${code}`, "auth");
      res.json({ message: "کد تایید ارسال شد", success: true });
    } catch (err: any) {
      res.status(400).json({ message: err.message });
    }
  });

  app.post("/api/auth/verify-otp", async (req, res) => {
    try {
      const { phone, code } = verifyOtpSchema.parse(req.body);
      const valid = await storage.verifyOtp(phone, code);
      if (!valid) {
        return res.status(400).json({ message: "کد تایید اشتباه است یا منقضی شده" });
      }

      let user = await storage.getUserByPhone(phone);
      if (!user) {
        user = await storage.createUser({ phone, role: "customer" });
      }

      req.session.userId = user.id;
      res.json({ user, success: true });
    } catch (err: any) {
      res.status(400).json({ message: err.message });
    }
  });

  app.get("/api/auth/me", async (req, res) => {
    if (!req.session.userId) {
      return res.status(401).json({ message: "Not authenticated" });
    }
    const user = await storage.getUser(req.session.userId);
    if (!user) {
      return res.status(401).json({ message: "User not found" });
    }
    res.json(user);
  });

  app.patch("/api/auth/profile", requireAuth, async (req, res) => {
    try {
      const { name, email, address, city, postalCode } = req.body;
      const updated = await storage.updateUser(req.session.userId!, {
        name, email, address, city, postalCode,
      });
      res.json(updated);
    } catch (err: any) {
      res.status(400).json({ message: err.message });
    }
  });

  app.post("/api/auth/logout", (req, res) => {
    req.session.destroy(() => {
      res.json({ success: true });
    });
  });

  app.get("/api/categories", async (_req, res) => {
    const cats = await storage.getVisibleCategories();
    res.json(cats);
  });

  app.get("/api/categories/all", async (_req, res) => {
    const cats = await storage.getCategories();
    res.json(cats);
  });

  app.get("/api/products", async (req, res) => {
    const { featured, category, exclude, limit } = req.query;
    if (featured === "true") {
      const p = await storage.getFeaturedProducts();
      return res.json(p);
    }
    if (category) {
      const catId = Number(category);
      if (!isNaN(catId)) {
        const p = await storage.getProductsByCategory(catId, exclude ? Number(exclude) : undefined);
        return res.json(p);
      }
      const cat = await storage.getCategoryBySlug(category as string);
      if (cat) {
        const p = await storage.getProductsByCategory(cat.id, exclude ? Number(exclude) : undefined);
        return res.json(p);
      }
    }
    const all = await storage.getProducts();
    if (limit) {
      return res.json(all.slice(0, Number(limit)));
    }
    res.json(all);
  });

  app.get("/api/products/:slug", async (req, res) => {
    const product = await storage.getProductBySlug(req.params.slug);
    if (!product) {
      return res.status(404).json({ message: "محصول یافت نشد" });
    }
    res.json(product);
  });

  app.get("/api/orders", requireAuth, async (req, res) => {
    const userOrders = await storage.getOrdersByUser(req.session.userId!);
    res.json(userOrders);
  });

  app.post("/api/orders", requireAuth, async (req, res) => {
    try {
      const { shippingAddress, shippingCity, shippingPostalCode, phone, notes, items } = req.body;

      if (!items || items.length === 0) {
        return res.status(400).json({ message: "سبد خرید خالی است" });
      }

      let total = 0;
      const productDetails: { product: any; quantity: number }[] = [];

      for (const item of items) {
        const product = await storage.getProductById(item.productId);
        if (!product) {
          return res.status(400).json({ message: `محصول ${item.productId} یافت نشد` });
        }
        if (!product.inStock || product.stockQuantity < item.quantity) {
          return res.status(400).json({ message: `موجودی ${product.name} کافی نیست` });
        }
        const price = Number(product.salePrice || product.price);
        total += price * item.quantity;
        productDetails.push({ product, quantity: item.quantity });
      }

      const order = await storage.createOrder({
        userId: req.session.userId!,
        total: total.toString(),
        shippingAddress,
        shippingCity,
        shippingPostalCode,
        phone,
        notes,
        status: "pending",
      });

      for (const detail of productDetails) {
        await storage.createOrderItem({
          orderId: order.id,
          productId: detail.product.id,
          productName: detail.product.name,
          productImage: detail.product.images?.[0] || null,
          quantity: detail.quantity,
          price: (detail.product.salePrice || detail.product.price).toString(),
        });

        await storage.updateProduct(detail.product.id, {
          stockQuantity: detail.product.stockQuantity - detail.quantity,
          inStock: detail.product.stockQuantity - detail.quantity > 0,
        });
      }

      res.json(order);
    } catch (err: any) {
      res.status(400).json({ message: err.message });
    }
  });

  app.get("/api/admin/products", requireAdmin, async (_req, res) => {
    const all = await storage.getProducts();
    res.json(all);
  });

  // Product CRUD disabled - products come from WooCommerce sync only
  app.post("/api/admin/products", requireAdmin, async (_req, res) => {
    res.status(403).json({ message: "محصولات فقط از طریق همگام‌سازی با ووکامرس قابل افزودن هستند" });
  });
  app.patch("/api/admin/products/:id", requireAdmin, async (_req, res) => {
    res.status(403).json({ message: "ویرایش محصولات فقط از طریق ووکامرس امکان‌پذیر است" });
  });
  app.delete("/api/admin/products/:id", requireAdmin, async (_req, res) => {
    res.status(403).json({ message: "حذف محصولات فقط از طریق ووکامرس امکان‌پذیر است" });
  });

  app.get("/api/admin/orders", requireAdmin, async (_req, res) => {
    const all = await storage.getAllOrders();
    res.json(all);
  });

  app.patch("/api/admin/orders/:id", requireAdmin, async (req, res) => {
    try {
      const order = await storage.updateOrderStatus(Number(req.params.id), req.body.status);
      res.json(order);
    } catch (err: any) {
      res.status(400).json({ message: err.message });
    }
  });

  app.get("/api/admin/settings", requireAdmin, async (_req, res) => {
    const all = await storage.getAllSettings();
    res.json(all);
  });

  app.post("/api/admin/settings", requireAdmin, async (req, res) => {
    try {
      const { wc_url, wc_key, wc_secret } = req.body;
      if (wc_url !== undefined) await storage.setSetting("wc_url", wc_url);
      if (wc_key !== undefined) await storage.setSetting("wc_key", wc_key);
      if (wc_secret !== undefined) await storage.setSetting("wc_secret", wc_secret);
      res.json({ success: true });
    } catch (err: any) {
      res.status(400).json({ message: err.message });
    }
  });

  app.get("/api/admin/sync-status", requireAdmin, async (_req, res) => {
    const lastSync = await storage.getSetting("last_sync_at");
    const syncResult = await storage.getSetting("last_sync_result");
    const productCount = (await storage.getProducts()).length;
    const categoryCount = (await storage.getCategories()).length;
    res.json({ lastSync, syncResult, productCount, categoryCount });
  });

  app.post("/api/admin/sync", requireAdmin, async (_req, res) => {
    try {
      const wcUrl = await storage.getSetting("wc_url");
      const wcKey = await storage.getSetting("wc_key");
      const wcSecret = await storage.getSetting("wc_secret");

      if (!wcUrl || !wcKey || !wcSecret) {
        return res.status(400).json({ message: "ابتدا تنظیمات ووکامرس را وارد کنید" });
      }

      const baseUrl = wcUrl.replace(/\/$/, "");
      const auth = Buffer.from(`${wcKey}:${wcSecret}`).toString("base64");
      const headers = { Authorization: `Basic ${auth}` };

      const wcFetch = async (url: string) => {
        const controller = new AbortController();
        const timeout = setTimeout(() => controller.abort(), 30000);
        try {
          const response = await fetch(url, { headers, signal: controller.signal });
          return response;
        } catch (err: any) {
          if (err.name === "AbortError") {
            throw new Error("اتصال به سایت ووکامرس بیش از ۳۰ ثانیه طول کشید. لطفاً آدرس سایت و اتصال اینترنت را بررسی کنید.");
          }
          if (err.cause?.code === "ENOTFOUND") {
            throw new Error(`آدرس سایت "${baseUrl}" پیدا نشد. لطفاً آدرس را بررسی کنید.`);
          }
          if (err.cause?.code === "ECONNREFUSED") {
            throw new Error(`اتصال به سایت "${baseUrl}" رد شد. سرور سایت ممکن است خاموش باشد.`);
          }
          if (err.cause?.code === "ETIMEDOUT" || err.cause?.code === "ENETUNREACH") {
            throw new Error(`سایت "${baseUrl}" قابل دسترسی نیست. ممکن است فایروال سرور دسترسی خارجی را مسدود کرده باشد.`);
          }
          throw new Error(`خطا در اتصال به ووکامرس: ${err.message}`);
        } finally {
          clearTimeout(timeout);
        }
      };

      let syncedCats = 0;
      let syncedProducts = 0;

      // Sync categories (all pages)
      let catPage = 1;
      let catHasMore = true;
      while (catHasMore) {
        const catRes = await wcFetch(`${baseUrl}/wp-json/wc/v3/products/categories?per_page=100&page=${catPage}`);
        if (!catRes.ok) {
          const errText = await catRes.text();
          log(`WC category fetch error: ${catRes.status} ${errText}`, "wc");
          if (catRes.status === 401) {
            throw new Error("کلیدهای API ووکامرس نامعتبر هستند. لطفاً Consumer Key و Consumer Secret را بررسی کنید.");
          }
          if (catRes.status === 404) {
            throw new Error("API ووکامرس در سایت شما فعال نیست یا آدرس سایت اشتباه است.");
          }
          throw new Error(`خطا در دریافت دسته‌بندی‌ها (کد ${catRes.status})`);
        }
        const wcCategories = await catRes.json();
        if (!Array.isArray(wcCategories) || wcCategories.length === 0) {
          catHasMore = false;
          break;
        }

        for (const wc of wcCategories) {
          if (wc.slug === "uncategorized") continue;
          const existing = await storage.getCategoryByWcId(wc.id);
          const catData = {
            name: wc.name,
            slug: wc.slug,
            image: wc.image?.src || null,
            wcId: wc.id,
            parentId: null as number | null,
          };
          if (wc.parent && wc.parent > 0) {
            const parentCat = await storage.getCategoryByWcId(wc.parent);
            if (parentCat) catData.parentId = parentCat.id;
          }

          if (existing) {
            await storage.updateCategory(existing.id, catData);
          } else {
            await storage.createCategory({ ...catData, visible: true, displayOrder: 0 });
          }
          syncedCats++;
        }
        catPage++;
      }

      // Sync products (all pages)
      let page = 1;
      let hasMore = true;
      while (hasMore) {
        const prodRes = await wcFetch(`${baseUrl}/wp-json/wc/v3/products?per_page=100&page=${page}&status=publish`);
        if (!prodRes.ok) {
          log(`WC product fetch error page ${page}: ${prodRes.status}`, "wc");
          break;
        }
        const wcProducts = await prodRes.json();
        if (!Array.isArray(wcProducts) || wcProducts.length === 0) {
          hasMore = false;
          break;
        }

        for (const wc of wcProducts) {
          const slug = wc.slug || `product-${wc.id}`;

          // Map WC category to local category
          let localCatId: number | null = null;
          if (wc.categories && wc.categories.length > 0) {
            const wcCatId = wc.categories[0].id;
            const localCat = await storage.getCategoryByWcId(wcCatId);
            if (localCat) localCatId = localCat.id;
          }

          // Parse attributes
          const attributes: Record<string, string> = {};
          if (wc.attributes && Array.isArray(wc.attributes)) {
            for (const attr of wc.attributes) {
              attributes[attr.name] = Array.isArray(attr.options) ? attr.options.join("، ") : String(attr.options);
            }
          }

          const productData = {
            name: wc.name,
            slug,
            description: wc.description?.replace(/<[^>]*>/g, "") || null,
            shortDescription: wc.short_description?.replace(/<[^>]*>/g, "") || null,
            price: wc.regular_price || wc.price || "0",
            salePrice: wc.sale_price && wc.sale_price !== "" ? wc.sale_price : null,
            sku: wc.sku || null,
            stockQuantity: wc.stock_quantity ?? 0,
            inStock: wc.in_stock ?? wc.stock_status === "instock",
            images: wc.images?.map((i: any) => i.src) || [],
            categoryId: localCatId,
            featured: wc.featured || false,
            attributes: Object.keys(attributes).length > 0 ? attributes : null,
            wcId: wc.id,
          };

          // Find by wcId first, then by slug
          const existing = await storage.getProductByWcId(wc.id) || await storage.getProductBySlug(slug);
          if (existing) {
            await storage.updateProduct(existing.id, productData);
          } else {
            await storage.createProduct(productData);
          }
          syncedProducts++;
        }
        page++;
      }

      const syncTime = new Date().toISOString();
      const resultMsg = `${syncedCats} دسته‌بندی و ${syncedProducts} محصول همگام‌سازی شد`;
      await storage.setSetting("last_sync_at", syncTime);
      await storage.setSetting("last_sync_result", resultMsg);

      log(`WooCommerce sync completed: ${resultMsg}`, "wc");
      res.json({ success: true, message: resultMsg, syncedCats, syncedProducts });
    } catch (err: any) {
      log(`WooCommerce sync error: ${err.message}`, "wc");
      await storage.setSetting("last_sync_at", new Date().toISOString());
      await storage.setSetting("last_sync_result", `خطا: ${err.message}`);
      res.status(500).json({ message: "خطا در همگام‌سازی: " + err.message });
    }
  });

  // Admin category display management
  app.get("/api/admin/categories", requireAdmin, async (_req, res) => {
    const cats = await storage.getCategories();
    res.json(cats);
  });

  app.patch("/api/admin/categories/:id", requireAdmin, async (req, res) => {
    try {
      const { visible, displayOrder, swapWithId } = req.body;
      if (swapWithId) {
        const cat = await storage.getCategories().then(cats => cats.find(c => c.id === Number(req.params.id)));
        const other = await storage.getCategories().then(cats => cats.find(c => c.id === swapWithId));
        if (cat && other) {
          await storage.updateCategory(cat.id, { displayOrder: other.displayOrder });
          await storage.updateCategory(other.id, { displayOrder: cat.displayOrder });
        }
        const updated = await storage.getCategories();
        return res.json(updated.find(c => c.id === Number(req.params.id)));
      }
      const cat = await storage.updateCategory(Number(req.params.id), {
        ...(visible !== undefined && { visible }),
        ...(displayOrder !== undefined && { displayOrder }),
      });
      res.json(cat);
    } catch (err: any) {
      res.status(400).json({ message: err.message });
    }
  });

  return httpServer;
}
